/* 
 * prodcons.c
 *
 * Моделює задачу виробників і споживачів на базі паралельних процесів
 * з використанням подільної пам'яті та семафорів System V, але 
 * не забезпечує взаємне виключення виробників і споживачів.
 * Ілюструє порядок застосування подільної пам'яті та семафорів System V.
 *
 */

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include "sem.h"

#define PROJNAME "prodcons"
/* Номери проектів, асоційованих із подільною пам'яттю
   і набором семафорів */
enum { SHM_PROJNUM = 1, SEM_PROJNUM =2 };

#define PERM 0600

/* Номери семафорів (потрібні, бо семафори System V створюються наборами, 
   в межах яких ідентифікуються за номерами) */
enum { EMPTYSEM_NUMBER = 0, FULLSEM_NUMBER, 
        PRODUCED_TOTAL_NUMBER, CONSUMED_TOTAL_NUMBER };

enum { 
        /* Розмір подільного буфера (буферного пулу) */
        BUFSIZE = 10000,
        /* Кількість виробників */
        NPRODUCERS = 2,
        /* Кількість споживачів */
        NCONSUMERS = 2,
        /* Кількість порцій даних, які треба виробити та спожити */
        NITEMS = 30000
};

/* Подільні змінні */
struct shared_vars {
    /* Номери наступної вільної та наступної заповненої позиції
       в буферному пулі */
    int in, out;
    /* Масив, у який виробники записують, скільки кожен із них виробив */
    int produced_ind[NPRODUCERS];
    /* Масив, у який споживачі записують, скільки вони спожили від кожного
       виробника */
    int consumed_ind[NPRODUCERS];
};

/* Покажчик на початок подільного сегмента пам'яті */
struct shared_vars *shmp;
/* Дескриптори подільної пам'яті та набору семафорів */
int shmd, semd;
/* Покажчик на початок буферного пулу */
int *buf;

/* Лічильник дочірніх процесів, які завершили роботу */
sig_atomic_t childs_died = 0;

int producer(int number), consumer();


int cleanup()
{
        int rval;

        rval = 0;
        /* Помічає для видалення сегмент подільної пам'яті */
        if (shmd > 0 && shmctl(shmd, IPC_RMID, NULL) < 0) {
                perror("shmctl()");
                rval = 1;
        }	
        /* Видаляє набір семафорів */
        if (semd > 0 && semctl(semd, 0, IPC_RMID) < 0) {
                perror("semctl(..., IPC_RMID)");
                rval = 1;
        }	
        return rval;
}

/* Обробник сигналу SIGCHLD */
void cleanup_child(int sig)
{
        wait(NULL);
        childs_died++;
}

int main()
{
        key_t shm_key, sem_key;   /* Ключі */
        struct sigaction act;
        int childs_created;     /* Лічильник дочірніх процесів */
        int i;

        /* Отримує ключ сегмента подільної пам'яті. */
        shm_key = ftok(PROJNAME, SHM_PROJNUM);
        if (shm_key == -1) {
                perror("shm ftok()");
                exit(EXIT_FAILURE);
        }
        /* Отримує ключ набору семафорів. */
        sem_key = ftok(PROJNAME, SEM_PROJNUM);
        if (sem_key == -1) {
                perror("sem ftok()");
                exit(EXIT_FAILURE);
        }	

        /* Створює/відкриває сегмент подільної пам'яті. */
        shmd = shmget(shm_key, BUFSIZE*sizeof(int) + 
                        sizeof(struct shared_vars), PERM | IPC_CREAT);
        if (shmd == -1) {
                perror("shmget()");
                exit(EXIT_FAILURE);
        }	
        /* Приєднує сегмент подільної пам'яті до свого віртуального
           адресного простору. */
        shmp = (struct shared_vars *) shmat(shmd, 0, 0);
        if (shmp == (void *) -1) {
                perror("shmat()");
                cleanup();
                exit(EXIT_FAILURE);
        }
        buf = (int *) (shmp + 1);

        /* Створює/відкриває набір із 4 семафорів. */
        semd = semget(sem_key, 4, PERM | IPC_CREAT);
        if (semd < 0) {
                perror("semget()");
                cleanup();
                exit(EXIT_FAILURE);
        }
        /* Ініціалізує початкові значення семафорів. */
        if (semctl(semd, EMPTYSEM_NUMBER, SETVAL, BUFSIZE) < 0 ||
                semctl(semd, FULLSEM_NUMBER, SETVAL, 0) < 0 ||
                semctl(semd, PRODUCED_TOTAL_NUMBER, SETVAL, NITEMS) < 0 ||
                semctl(semd, CONSUMED_TOTAL_NUMBER, SETVAL, NITEMS) < 0) {
                perror("semctl(..., SETVAL, ...)");
                cleanup();
                exit(EXIT_FAILURE);
        }

        /* Реєструє обробник сигналу SIGCHLD. */
        memset(&act, 0, sizeof(act));
        act.sa_handler = &cleanup_child;
        if (sigaction(SIGCHLD, &act, NULL) != 0) {
                perror("sigaction()");
                cleanup();
                exit(EXIT_FAILURE);
        }

        shmp->in = 0;
        shmp->out = 0;
        childs_created = 0;
        /* Створює/запускає виробників */
        for (i = 0; i < NPRODUCERS; i++) {
                switch(fork()) {
                case -1:
                        perror("fork()");
                        break;
                case 0:
                        /* Дочірній процес (виробник) */
                        if (producer(i) != 0)
                                exit(EXIT_FAILURE);
                        exit(EXIT_SUCCESS);
                default:
                        childs_created++;
	       }
        }
        /* Створює/запускає споживачів. */
        for (i = 0; i < NCONSUMERS; i++) {
                switch(fork()) {
                case -1:
                        perror("fork()");
                        break;
                case 0:
                        /* Дочірній процес (споживач) */
                        if (consumer() != 0)
                                exit(EXIT_FAILURE);
                        exit(EXIT_SUCCESS);
                default:
                        childs_created++;
                }
        }

        /* Установлює ігнорування сигналу SIGCHLD. */
        act.sa_handler = SIG_IGN;
        while (sigaction(SIGCHLD, &act, NULL) != 0) {
                if (errno != EINTR) {
                        perror("sigaction()");
                        cleanup();
                        exit(EXIT_FAILURE);
                }
        }
        /* Чекає завершення роботи всіх нащадків. */  
        while (childs_died < childs_created) {
                wait(NULL);
                childs_died++;
        }

        /* Аналізує результати роботи */
        {
                int consumed, produced;
                int errnum = 0;

                for (i = 0; i < NPRODUCERS; i++) {
                        printf("produced_ind[%d] = %d\t"
                                "consumed_ind[%d] = %d",
                                i, shmp->produced_ind[i],
                                i, shmp->consumed_ind[i]);
                        if (shmp->produced_ind[i] !=
                                        shmp->consumed_ind[i]) {
                                errnum++;
                                printf("\tERROR\n");
                        } else
                                printf("\tOK\n");
                }
                produced = semctl(semd, PRODUCED_TOTAL_NUMBER, GETVAL);
                if (produced == -1) {
                        perror("semctl(semd, PRODUCED_TOTAL_NUMBER,"
                                                        " GETVAL)");
                        cleanup();
                        exit(EXIT_FAILURE);
                }
                printf("Produced together: %d\n", NITEMS - produced);
                consumed = semctl(semd, CONSUMED_TOTAL_NUMBER, GETVAL);
                if (consumed == -1) {
                        perror("semctl(semd, CONSUMED_TOTAL_NUMBER,"
                                                        " GETVAL)");
                        cleanup();
                        exit(EXIT_FAILURE);
                }
                printf("Consumed together: %d\n", NITEMS - consumed);

                if (errnum > 0)
                        printf("There are some errors\n");
                else
                        printf("There are no errors\n");    
        }	

        cleanup();
        exit(EXIT_SUCCESS);
}

/* Функція виробника */
int producer(int number)
{
        for (;;) {
                /* Перевіряє, чи треба ще щось виробляти. */
                if (sem_trydown(semd, PRODUCED_TOTAL_NUMBER) != 0) {
                        if (errno == EAGAIN)
                                return 0;
                        else {
                                perror("sem_trydown(semd,"
                                        " PRODUCED_TOTAL_NUMBER)");
                                return 1;
                        }
                }
                /* Перевіряє, чи є вільні буфери (якщо немає,
                   блокується). */
                if (sem_down(semd, EMPTYSEM_NUMBER) != 0) {
                        perror("sem_down(semd, EMPTYSEM_NUMBER)");
                        return 1;
                }
                /* Виробляє порцію даних (просто зберігає в першій вільній
                   комірці буферного пулу свій номер). */
                buf[shmp->in] = number;
                shmp->in++;
                shmp->in %= BUFSIZE;
                shmp->produced_ind[number]++;
                /* Збільшує лічильник заповнених буферів. */
                if (sem_up(semd, FULLSEM_NUMBER) != 0) {
                        perror("sem_up(semd, FULLSEM_NUMBER)");
                        return 1;
                }
        }
}

/* Функція споживача */
int consumer()
{
        for (;;) {
                /* Перевіряє, чи треба ще щось споживати. */
                if (sem_trydown(semd, CONSUMED_TOTAL_NUMBER) != 0) {
                        if (errno == EAGAIN)
                                return 0;
                        else {
                                perror("sem_trydown(semd,"
                                        " CONSUMED_TOTAL_NUMBER)");
                                return 1;
                        }
                }
                /* Перевіряє, чи є заповнені буфери (якщо немає,
                   блокується). */
                if (sem_down(semd, FULLSEM_NUMBER) < 0) {
                        perror("sem_down(semd, EMPTYSEM_NUMBER)");
                        return 1;
                }
                /* Споживає порцію даних. */
                shmp->consumed_ind[buf[shmp->out]]++;
                shmp->out++;
                shmp->out %= BUFSIZE;
                /* Збільшує лічильник вільних буферів. */
                if (sem_up(semd, EMPTYSEM_NUMBER) < 0) {
                        perror("sem_up(semd, EMPTYSEM_NUMBER)");
                        return 1;
                }
        }
}
